from flask import Flask, render_template, request, redirect, url_for, session, send_from_directory, flash
import os
from werkzeug.utils import secure_filename
from utils.ocr_utils import extract_text_from_image
from datetime import datetime

app = Flask(__name__)
app.secret_key = 'your_secret_key'
app.config['UPLOAD_FOLDER'] = 'uploads'
app.config['RESULT_FOLDER'] = 'results'

# Allowed extensions for upload
ALLOWED_EXTENSIONS = {'png', 'jpg', 'jpeg', 'pdf'}

# Dummy in-memory user store
users = {'test': 'test123'}

# ---------------- Helper Functions ---------------- #
def allowed_file(filename):
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS

# ---------------- Routes ---------------- #

@app.route('/')
def home():
    if 'username' in session:
        return redirect(url_for('dashboard'))
    return redirect(url_for('login'))

@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        if username in users and users[username] == password:
            session['username'] = username
            return redirect(url_for('dashboard'))
        else:
            return render_template('login.html', error="Invalid credentials")
    return render_template('login.html')

@app.route('/register', methods=['GET', 'POST'])
def register():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        if username in users:
            return render_template('register.html', error="Username already exists")
        users[username] = password
        session['username'] = username
        return redirect(url_for('dashboard'))
    return render_template('register.html')

@app.route('/dashboard')
def dashboard():
    if 'username' in session:
        now = datetime.now()
        current_date = now.strftime("%A, %d %B %Y")
        current_time = now.strftime("%I:%M %p")
        return render_template('dashboard.html', username=session['username'], current_date=current_date, current_time=current_time)
    return redirect(url_for('login'))

@app.route('/upload', methods=['GET', 'POST'])
def upload():
    if 'username' not in session:
        return redirect(url_for('login'))

    if request.method == 'POST':
        if 'file' not in request.files:
            return 'No file uploaded.'

        file = request.files['file']
        if file.filename == '':
            return 'No selected file.'

        if not allowed_file(file.filename):
            return 'Unsupported file type. Only images and PDFs allowed.'

        filename = secure_filename(file.filename)
        upload_path = os.path.join(app.config['UPLOAD_FOLDER'], filename)
        file.save(upload_path)

        # OCR processing
        text = extract_text_from_image(upload_path)

        result_filename = filename + '.txt'
        result_path = os.path.join(app.config['RESULT_FOLDER'], result_filename)
        with open(result_path, 'w', encoding='utf-8') as f:
            f.write(text)

        return render_template('result.html', filename=filename, text=text, result_file=result_filename)

    return render_template('upload.html')

@app.route('/downloads')
def downloads():
    if 'username' not in session:
        return redirect(url_for('login'))

    # Get the list of result files
    files = os.listdir(app.config['RESULT_FOLDER'])

    # If files exist, show the most recent one
    if files:
        latest_file = max(
            [os.path.join(app.config['RESULT_FOLDER'], f) for f in files],
            key=os.path.getctime
        )
        result_filename = os.path.basename(latest_file)
        return render_template('download.html', result_file=result_filename)
    else:
        return render_template('download.html', result_file=None)

@app.route('/download/<filename>')
def download_result(filename):
    if 'username' not in session:
        return redirect(url_for('login'))
    
    result_path = os.path.join(app.config['RESULT_FOLDER'], filename)
    if os.path.exists(result_path):
        return send_from_directory(app.config['RESULT_FOLDER'], filename, as_attachment=True)
    else:
        return "File not found.", 404


@app.route('/logout')
def logout():
    session.pop('username', None)
    return render_template('logout.html')

# ---------------- Startup ---------------- #
if __name__ == '__main__':
    os.makedirs(app.config['UPLOAD_FOLDER'], exist_ok=True)
    os.makedirs(app.config['RESULT_FOLDER'], exist_ok=True)
    app.run(debug=True)
